<?xml version="1.0" encoding="iso-8859-1"?>
<!-- Copyright (c) 2002-2005 DeltaXML Ltd. All rights reserved -->
<!-- Use subject to the conditions in the included file: LICENSE.bsd -->
<!-- $Id: deltaxml-folding-html.xsl 4527 2008-06-10 16:35:27Z nigelw $ -->
<!--
This stylesheet converts a delta file, produced by deltaXML programs,
into HTML that can be displayed in a browser and gives a table structure
for the changed data. It may run into trouble for large delta files because
the HTML tables may be too large or complex to display in browsers.
-->

<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:deltaxml="http://www.deltaxml.com/ns/well-formed-delta-v1">

  <!-- change indent to "yes" for pretty-printed output where white space is not critical,
       or to "no" if white space is critical and you do not want any added -->
  <xsl:output method="xml" indent="yes" version="1.0"
    
    encoding="iso-8859-1" media-type="text/html"/>

  <!-- When set to 'no' key-information for modified elements is not shown,
       it is shown when set to 'yes' -->
  <xsl:param name="display-keys" select="'yes'"/>
  
  <!-- Childless nodes smaller than this number of characters are not foldable -->
  <xsl:param name="no-fold-size" select="50"/>
  <!-- this param contains xpaths to attributes that should remain visible even
       when the element's attributes are folded -->
  <xsl:param name="sticky-atts"/>
  <!-- this param is used to determine whether unchhaned elements should be initially collapsed -->
  <xsl:param name="collapseUnchanged" select="'yes'"/>
  
  <xsl:template match="/">
   <html>
    <head>
      <title>Delta output example</title>
      <style type="text/css">
      <xsl:text>
        /* place a border around the delta file and indent on left to allow for +/- button */ 
        div#deltafile { 
          border: 2px solid black;
          padding: 5px 5px 5px 25px;
        }
        /* styling for different data types */
        div.delete, span.delete, span.old-PCDATA {
          color: red;
          text-decoration: line-through;
          font-style: normal;
        }
        div.add, span.add, span.new-PCDATA {
          color: green;
          text-decoration: underline;
          font-style: normal;
        }
        span.comment {
          color: black; 
        }
        span.modify, span.modify-PCDATA {
          color: blue; 
          font-style: italic;
        }
        div.unchanged, span.unchanged{ 
          color: black; 
          font-style: normal;
        }
        /* styling and positioning for +/- buttons */
        a.fold { 
          position: relative;
          top: 5px; /* added to align the button with the element */
          float: left;
          color: black;
          font-family: monospace;
          font-weight: bold;
          font-style: normal;
          font-size: small;
          background: #ddd;
          width: 10px;
          height: 10px;
          border: 1px outset black;
          text-align: center;
          line-height: 11px;
          margin-left: -20px
        }
        a.fold:hover {
          cursor: pointer;
        }
        /* gives 'pressed button' effect */
        a.fold:active {
          border-style: inset;
        }
        a.foldall {
          background: #ddd;
          border: 1px outset black;
          padding: 2px;
        }
        a.foldall:hover {
          cursor: pointer;
        }
        a.foldall:active {
          border-style: inset;
        }
        
        /* hides collapsed(folded children) and hidden(unchanged atts/children) */
        .collapsed, .hidden { 
          display:none; 
        }
        div.anchor-parent div.anchor-parent{
          margin-left: 20px;
        }
        /* provides vertical spacing between lines */
        div#deltafile span {
          line-height: 1.5em;
        }
      </xsl:text>
    </style>
    <script type="text/javascript">
      <xsl:text>
        function fold(obj) {
          //switch anchor text between '+' and '-'
          obj.innerHTML= (obj.innerHTML == '-') ? '+' : '-';
          //switch anchor title between 'expand' and 'collapse'
          obj.title= (obj.title == 'collapse') ? 'expand' : 'collapse';
  
          //the next two lines get all 'span' siblings of the a tag that was 
          //clicked and then pick the first one. This is the one whose children
          //must be hidden. We also don't select text nodes this way.
          var siblings= obj.parentNode.getElementsByTagName('span');
          var tmp= siblings[0];
          
          //go through all children of the node selected and test their className
          //if 'expanded' or 'collapsed', switch, otherwise ignore it
          for (i=0; i != tmp.childNodes.length; i++)
          {
            var name= new String(tmp.childNodes[i].className);
            if (name == 'expanded' || name == 'collapsed') {
              tmp.childNodes[i].className = (tmp.childNodes[i].className == 'expanded') ? 'collapsed' : 'expanded';
            } else {
              //no change to the name if it isn't one of 'expanded' or 'collapsed'
            }
          }
        }
        function foldall(obj) {
          var collapse= (obj.innerHTML == 'Collapse all');
          //switch text between 'collapse all' and 'expand all'
          obj.innerHTML= collapse ? 'Expand all' : 'Collapse all';
          
          //now get all anchor tags and fold them
          var anchors= document.getElementsByTagName("a");
          for (i=0; i != anchors.length; i++) {
            if (anchors[i].className == 'fold') {
              anchors[i].innerHTML= collapse ? '+' : '-';
              anchors[i].title= collapse? 'expand' : 'collapse';
              var siblings= anchors[i].parentNode.getElementsByTagName('span');
              var span= siblings[0];
              
              for (j=0; j != span.childNodes.length; j++){
                var name= new String(span.childNodes[j].className);
                if (name == 'expanded' || name == 'collapsed') {
                  if(span.childNodes[j].id == 'content-span') {
                    span.childNodes[j].className= collapse ? 'collapsed' : 'expanded';
                  } else {
                    span.childNodes[j].className= collapse ? 'expanded' : 'collapsed';
                  }
                } else {
                  //no change
                }
              }
            }
          }
        }
        
      </xsl:text>
    </script>
    </head>
    <body>
      <h1>Display of changes found using DeltaXML</h1>
      <p>This file displays the results of comparing two files. The delta file generated by DeltaXML is an XML
      file. It has been translated into the HTML seen here by an XSL style sheet, so all the display styles,
      colours etc. can easily be changed by modifying the XSL stylesheet.
      This stylesheet works best with a 'full' delta output, which includes unchanged data as well as changed data.</p>
      <p>Data that has been modified is shown <span class="modify">like this</span>. Data that has been deleted is shown <span class="delete">like this</span>.
      Data that is added is shown <span class="add">like this</span>.</p>
      <p>To collapse a node's children click the '<strong>-</strong>' button. To expand it again, click '<strong>+</strong>'. Unchanged elements are collapsed by default.</p>
      <p>This page is best viewed with Cascading Style Sheets and JavaScript both enabled.</p>
      <p>For further details of DeltaXML see <a href="http://www.deltaxml.com">http://www.deltaxml.com</a></p>
      <!--<p><a class="foldall" onclick="foldall(this)">Collapse all</a></p>-->
      <div id="deltafile">
        <xsl:apply-templates/>
      </div>
    </body>
   </html>
  </xsl:template>
  
  <!-- This matches all elements other than PCDATAmodify blocks -->
  <xsl:template match="*">
    <!-- determine what type of element this is: modified, deleted, added or unchanged -->
    <xsl:variable name="delta-type">
      <xsl:choose>
        <xsl:when test="./ancestor-or-self::*[@deltaxml:delta][1][contains(@deltaxml:delta, 'modify')]">
          <xsl:value-of select="'modify'"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="./ancestor-or-self::*[@deltaxml:delta][1]/@deltaxml:delta"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    
    <!-- determine the size of the element -->
    <xsl:variable name="size">
      <xsl:call-template name="element-size">
        <xsl:with-param name="element" select="."/>
      </xsl:call-template>
    </xsl:variable>
    
    <!-- start the div that will be used for the indenting -->
    <div class="anchor-parent">
      <!-- a fold button is added if any of the following is true:
           1) the element has children
           2) the size of the element is greater than the $no-fold-size value (not counting sticky attributes)
           3) the delta-type is unchanged and the element has non-sticky attributes -->
      <xsl:if test="node()  or ($size > $no-fold-size) or($delta-type='unchanged' and @*[not(starts-with(name(), 'deltaxml:delta'))])">
        <a class="fold" onclick="fold(this)">
          <xsl:choose>
            <xsl:when test="$delta-type='unchanged' and $collapseUnchanged='yes'">
              <xsl:attribute name="title"><xsl:text>expand</xsl:text></xsl:attribute>
              <xsl:text>+</xsl:text>
            </xsl:when>
            <xsl:otherwise>
              <xsl:attribute name="title"><xsl:text>collapse</xsl:text></xsl:attribute>
              <xsl:text>-</xsl:text>
            </xsl:otherwise>
          </xsl:choose>
        </a>
      </xsl:if>
      
      <!-- place the element inside a span with a classname that tells up what delta-type it is -->
      <span class="{$delta-type}">
        <!-- start the element with a '<' and the element name -->
        &lt;<xsl:value-of select="name(.)"/>
        <!-- if the element has attributes that are not deltaxml:delta attributes, we need to process them now -->
        <xsl:if test="@*[not(starts-with(name(), 'deltaxml:delta'))]">
          
          <!-- first let's write the unchanged attributes, these are the ones with a name that doesn't start with 'deltaxml' -->
          <xsl:apply-templates select="@*[not(starts-with(name(), 'deltaxml'))]"/>
          <!-- now we need to process, modified, deleted and added attributes. They must be processed together. -->
          <xsl:call-template name="write-new-old-changed-attributes">
            <xsl:with-param name="old-attributes" select="@deltaxml:old-attributes"/>
            <xsl:with-param name="new-attributes" select="@deltaxml:new-attributes"/>
          </xsl:call-template>
          
          <!-- TODO this span should only be included if there are no 'non-sticky' attributes -->
          <span>
            <xsl:attribute name="class">
              <xsl:choose>
                <xsl:when test="$delta-type='unchanged' and $collapseUnchanged='yes'">
                  <xsl:text>expanded</xsl:text>
                </xsl:when>
                <xsl:otherwise>
                  <xsl:text>collapsed</xsl:text>
                </xsl:otherwise>
              </xsl:choose>
            </xsl:attribute>
            <span class="comment">
              <!-- the span's contents, this indicates there is more that can be viewed here -->
              <xsl:text> ... </xsl:text>
            </span>
          </span>
        </xsl:if>
        
        <!-- we now need to test for child nodes because this will determine whether we are in a self-closing tag or not -->
        <xsl:choose>
          <!-- the element does have child nodes, close this start tag with a '>' and process those nodes -->
          <xsl:when test="node()">&gt;
            <!-- enclose the children in a span so that we can fold it. -->
            <span id="content-span">
                <xsl:attribute name="class">
                  <!-- the initial class of the span containing the content depends on
                       what delta-type we have. Unchanged elements start off with their content collapsed -->
                  <xsl:choose>
                    <xsl:when test="$delta-type='unchanged' and $collapseUnchanged='yes'">
                      <xsl:text>collapsed</xsl:text>
                    </xsl:when>
                    <xsl:otherwise>
                      <xsl:text>expanded</xsl:text>
                    </xsl:otherwise>
                  </xsl:choose>
                </xsl:attribute>
              <!-- now process the child nodes -->
              <xsl:apply-templates select="node()"/>
            </span>
            <!-- lets write a span to be shown when the node contents are hidden.
                 Again the initial class of the span depends on delta-type -->
            <span>
              <xsl:attribute name="class">
                <xsl:choose>
                  <xsl:when test="$delta-type='unchanged' and $collapseUnchanged='yes'">
                    <xsl:text>expanded</xsl:text>
                  </xsl:when>
                  <xsl:otherwise>
                    <xsl:text>collapsed</xsl:text>
                  </xsl:otherwise>
                </xsl:choose>
              </xsl:attribute>
              <span class="comment">
                <!-- the span's contents, this indicates there is more that can be viewed here -->
                <xsl:text> ... </xsl:text>
              </span>
            </span>
            <!-- now write the closing tag for the element -->
            &lt;/<xsl:value-of select="name(.)"/>&gt;
          </xsl:when>
          <!-- the element does not have child nodes, this is therefore a self-closing tag, finish it with a '/>' -->
          <xsl:otherwise>/&gt;</xsl:otherwise>
        </xsl:choose>
      </span>
    </div>
  </xsl:template>
  
  <!-- This template deals with deltaxml:exchange elements. Simple process the node's grandchildren -->
  <xsl:template match="deltaxml:exchange">
    <xsl:apply-templates select="./deltaxml:old/*|./deltaxml:old/text()"/>
    <xsl:apply-templates select="./deltaxml:new/*|./deltaxml:new/text()"/>
  </xsl:template>
  
  <!-- Process any modified PCDATA -->
  <xsl:template match="deltaxml:PCDATAmodify">
    <xsl:variable name="old-data" select="deltaxml:PCDATAold/text()"/>
    <xsl:variable name="new-data" select="deltaxml:PCDATAnew/text()"/>
    <xsl:choose>
      <xsl:when test="string-length(normalize-space($old-data))>0
                 or string-length(normalize-space($new-data))>0">
        <!-- test that there is something in the string before outputting a span -->
        <xsl:if test="string-length(normalize-space($old-data))>0">
          <span class="delete"><xsl:value-of select="$old-data"/></span>
        </xsl:if>
        <xsl:if test="string-length(normalize-space($new-data))>0">
          <span class="add"><xsl:value-of select="$new-data"/></span>
        </xsl:if>
      </xsl:when>
      <xsl:otherwise>
        <span class="modify">...whitespace changes in PCDATA...</span>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>
  
  <!-- This match is for text() nodes -->
  <xsl:template match="text()">
    <!-- text nodes within a deltaxml:exchange block need special treatment -->
    <xsl:choose>
      <xsl:when test="./parent::*[name() = 'deltaxml:new'] | ./parent::*[name() = 'deltaxml:old']">
        <div class="anchor-parent">
          <xsl:if test="string-length(.) > $no-fold-size">
            <a class="fold" onclick="fold(this)" title="collapse">-</a>
          </xsl:if>
          <span>
            <span id="content-span" class="expanded">
              <span>
                <xsl:attribute name="class">
                  <xsl:choose>
                    <xsl:when test="./parent::*[name() = 'deltaxml:new']">
                      <xsl:value-of select="'add'"/>
                    </xsl:when>
                    <xsl:otherwise>
                      <xsl:value-of select="'delete'"/>
                    </xsl:otherwise>
                  </xsl:choose>
                </xsl:attribute>
                <xsl:value-of select="."/>  
              </span>
            </span>
            <span class="collapsed">
              <span class="comment">
                <xsl:text>PCDATA ...</xsl:text>
              </span>
            </span>
          </span>
        </div>
      </xsl:when>
      <xsl:otherwise>
        <span>
          <xsl:attribute name="class">
            <xsl:choose>
              <!-- first, if parent is deltaxml:old or deltaxml:new, set val appropriately -->
              <!--<xsl:when test="./parent::*[name() = 'deltaxml:old']">
                <xsl:value-of select="'delete'"/>
                </xsl:when>
                <xsl:when test="./parent::*[name() = 'deltaxml:new']">
                <xsl:value-of select="'add'"/>
                </xsl:when>-->
              <xsl:when test="./ancestor-or-self::*[@deltaxml:delta][1][contains(@deltaxml:delta, 'modify')]">
                <xsl:value-of select="'unchanged'"/>
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="./ancestor-or-self::*[@deltaxml:delta][1]/@deltaxml:delta"/>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:attribute>
          <xsl:value-of select="."/>
        </span>
      </xsl:otherwise>
    </xsl:choose>
    
  </xsl:template>
  
  <!-- This template will process attributes with no deltaxml: prefix. -->
  <xsl:template match="@*">
    
    <!-- find the delta-type of the parent element. If it is modify, these attributes will be unchanged -->
    <xsl:variable name="delta-type">
      <xsl:choose>
        <xsl:when test="./ancestor-or-self::*[@deltaxml:delta][1][contains(@deltaxml:delta, 'modify')]">
          <xsl:value-of select="'unchanged'"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="./ancestor-or-self::*[@deltaxml:delta][1]/@deltaxml:delta"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    
    <!-- surround the attribute with a span whose class is either collapsed or expanded
         depending on the delta-type of the parent element -->
    <span id="content-span">
      
        <xsl:attribute name="class">
          <xsl:choose>
            <!-- only for those attributes whose parent is unchanged. Doesn't include modified parents -->
            <xsl:when test="./ancestor-or-self::*[@deltaxml:delta][1]/@deltaxml:delta='unchanged' and $collapseUnchanged='yes'">
              <xsl:text>collapsed</xsl:text>
            </xsl:when>
            <xsl:otherwise>
              <xsl:text>expanded</xsl:text>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:attribute>

      <span class="{$delta-type}">
        <xsl:text> </xsl:text>
        <xsl:value-of select="name(.)"/>
        <xsl:text>=&quot;</xsl:text>
        <xsl:value-of select="."/>
        <xsl:text>&quot;</xsl:text>
      </span>
    </span>
  </xsl:template>
  
  <!-- this template is called after writing out unchanged attributes
       It must determine which attributes are added, deleted or modified -->
  <xsl:template name="write-new-old-changed-attributes">
    <xsl:param name="old-attributes"/>
    <xsl:param name="new-attributes"/>
    
    <xsl:variable name="all-old-attribute-names">
      <xsl:call-template name="all-attribute-names">
        <xsl:with-param name="delta-attribute-string"
          select="$old-attributes"/>
      </xsl:call-template>
    </xsl:variable>
    <xsl:variable name="all-new-attribute-names">
      <xsl:call-template name="all-attribute-names">
        <xsl:with-param name="delta-attribute-string"
          select="$new-attributes"/>
      </xsl:call-template>
    </xsl:variable>
    <xsl:variable name="deleted-attribute-names">
      <xsl:call-template name="nmtokens-set-difference">
        <xsl:with-param name="first-string" select="$all-old-attribute-names"/>
        <xsl:with-param name="second-string" select="$all-new-attribute-names"/>
      </xsl:call-template>
    </xsl:variable>
    <xsl:variable name="added-attribute-names">
      <xsl:call-template name="nmtokens-set-difference">
        <xsl:with-param name="first-string" select="$all-new-attribute-names"/>
        <xsl:with-param name="second-string" select="$all-old-attribute-names"/>
      </xsl:call-template>
    </xsl:variable>
    <xsl:variable name="changed-attribute-names">
      <xsl:call-template name="nmtokens-set-intersection">
        <xsl:with-param name="first-string" select="$all-old-attribute-names"/>
        <xsl:with-param name="second-string" select="$all-new-attribute-names"/>
      </xsl:call-template>
    </xsl:variable>
    
    <!-- now we know what's what, let's write them out -->
    <!-- start with old attributes -->
    <xsl:if test="string-length(normalize-space($deleted-attribute-names)) > 0">
      <xsl:call-template name="write-attribute-values">
        <xsl:with-param name="attribute-names" select="$deleted-attribute-names"/>
        <xsl:with-param name="delta-attribute-string" select="$old-attributes"/>
        <xsl:with-param name="delta-type" select="'delete'"/>
      </xsl:call-template>
    </xsl:if>
    
    <!-- now write out new attributes -->
    <xsl:if test="string-length(normalize-space($added-attribute-names)) > 0">
      <xsl:call-template name="write-attribute-values">
        <xsl:with-param name="attribute-names" select="$added-attribute-names"/>
        <xsl:with-param name="delta-attribute-string" select="$new-attributes"/>
        <xsl:with-param name="delta-type" select="'add'"/>
      </xsl:call-template>
    </xsl:if>
    
    <!-- now write out changed attributes -->
    <xsl:if test="string-length(normalize-space($changed-attribute-names)) > 0">
      <xsl:call-template name="write-changed-attribute-values">
        <xsl:with-param name="attribute-names" select="$changed-attribute-names"/>
        <xsl:with-param name="new-delta-attribute-string" select="$new-attributes"/>
        <xsl:with-param name="old-delta-attribute-string" select="$old-attributes"/>
      </xsl:call-template>
    </xsl:if>
  </xsl:template>
  
  <!-- returns a number representing the size in characters of an element -->
  <xsl:template name="element-size">
    <xsl:param name="element"/>
    <!-- the size of the element name itself plus '<', '>' and '/' -->
    <xsl:variable name="element-name">
      <xsl:choose>
        <xsl:when test="node()">
          <xsl:value-of select="string-length(name()) * 2 + 5"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="string-length(name()) + 3"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    
    <!-- TODO should sticky attributes count towards the size of the element? -->
    <xsl:variable name="attributes">
      <xsl:for-each select="@*[not(starts-with(name(), 'deltaxml'))]">
        <xsl:text> </xsl:text>
        <xsl:value-of select="name()"/>
        <xsl:text>=&quot;</xsl:text>
        <xsl:value-of select="."/>
        <xsl:text>&quot;</xsl:text>
      </xsl:for-each>
      <xsl:call-template name="write-new-old-changed-attributes">
        <xsl:with-param name="old-attributes" select="@deltaxml:old-attributes"/>
        <xsl:with-param name="new-attributes" select="@deltaxml:new-attributes"/>
      </xsl:call-template>
    </xsl:variable>
    
    <xsl:value-of select="$element-name + string-length($attributes)"/>
  </xsl:template>
  
  <!-- THE FOLLOWING TEMPLATES ARE UNCHANGED FROM THE ORIGINAL STYLESHEET -->
  
  <!-- templates for writing out deleted, added and changed attributes -->
  
  <!-- Returns a string of all the attribute names in the delta-attribute-string -->
  <xsl:template name="all-attribute-names">
    <xsl:param name="delta-attribute-string"/>
    <!-- find the name of the first attribute -->
    <xsl:variable name="first-attribute-name"
      select="substring-before($delta-attribute-string,'=')"/>
    <xsl:if test="string-length($first-attribute-name)>0">
      <xsl:variable name="attribute-value-delimiter"
        select="substring(substring-after($delta-attribute-string,'='),1,1)"/>
      <xsl:variable name="attribute-value-and-rest"
        select="substring-after($delta-attribute-string, $attribute-value-delimiter)"/>
      <xsl:variable name="other-attributes">
        <xsl:call-template name="all-attribute-names">
          <xsl:with-param name="delta-attribute-string"
            select="substring-after($attribute-value-and-rest, $attribute-value-delimiter)"/>
        </xsl:call-template>
      </xsl:variable>
      <xsl:value-of select="concat($first-attribute-name, ' ', $other-attributes)"/>
    </xsl:if>
  </xsl:template>
  
  <!-- Writes out the name and value of attributes whose names are in attribute-names,
  taking the values from delta-attribute-string -->
  <xsl:template name="write-attribute-values">
    <xsl:param name="delta-attribute-string"/>
    <xsl:param name="attribute-names"/>
    <xsl:param name="delta-type"/>
    <!-- find the name of the first attribute -->
    <xsl:variable name="first-attribute-name"
      select="substring-before($attribute-names,' ')"/>
    <xsl:if test="string-length($first-attribute-name)>0">
      <xsl:variable name="first-attribute-value">
        <xsl:call-template name="find-attribute-value">
          <xsl:with-param name="delta-attribute-string" select="$delta-attribute-string"/>
          <xsl:with-param name="attribute-name" select="$first-attribute-name"/>
        </xsl:call-template>
      </xsl:variable>
      <span id="content-span" class="expanded">
        <span>
          <xsl:attribute name="class">
            <xsl:value-of select="$delta-type"/>
          </xsl:attribute>
          <xsl:text> </xsl:text>
          <xsl:value-of select="$first-attribute-name"/>
          <xsl:text>=&quot;</xsl:text>
          <xsl:value-of select="$first-attribute-value"/>
          <xsl:text>&quot;</xsl:text>
        </span>
      </span>
      <xsl:variable name="other-attribute-names"
        select="substring-after($attribute-names,' ')"/>
      <xsl:call-template name="write-attribute-values">
        <xsl:with-param name="delta-attribute-string" select="$delta-attribute-string"/>
        <xsl:with-param name="attribute-names" select="$other-attribute-names"/>
		<xsl:with-param name="delta-type" select="$delta-type"/>
      </xsl:call-template>
    </xsl:if>
  </xsl:template>
  
  <!-- Writes out the name and changed value of attributes whose names are in attribute-names,
  taking the values from old-delta-attribute-string and new-delta-attribute-string -->
  <xsl:template name="write-changed-attribute-values">
    <xsl:param name="old-delta-attribute-string"/>
    <xsl:param name="new-delta-attribute-string"/>
    <xsl:param name="attribute-names"/>
    <!-- find the name of the first attribute -->
    <xsl:variable name="first-attribute-name"
      select="substring-before($attribute-names,' ')"/>
    <xsl:if test="string-length($first-attribute-name)>0">
      <xsl:variable name="first-attribute-old-value">
        <xsl:call-template name="find-attribute-value">
          <xsl:with-param name="delta-attribute-string" select="$old-delta-attribute-string"/>
          <xsl:with-param name="attribute-name" select="$first-attribute-name"/>
        </xsl:call-template>
      </xsl:variable>
      <xsl:variable name="first-attribute-new-value">
        <xsl:call-template name="find-attribute-value">
          <xsl:with-param name="delta-attribute-string" select="$new-delta-attribute-string"/>
          <xsl:with-param name="attribute-name" select="$first-attribute-name"/>
        </xsl:call-template>
      </xsl:variable>
      <span id="content-span" class="expanded">
        <xsl:text> </xsl:text>
        <xsl:value-of select="$first-attribute-name"/>
        <xsl:text>=&quot;</xsl:text>
        <span class="delete"><xsl:value-of select="$first-attribute-old-value"/></span>
        <span class="add"><xsl:value-of select="$first-attribute-new-value"/></span>
        <xsl:text>&quot;</xsl:text>
      </span>
      <xsl:variable name="other-attribute-names"
        select="substring-after($attribute-names,' ')"/>
      <xsl:call-template name="write-changed-attribute-values">
        <xsl:with-param name="old-delta-attribute-string" select="$old-delta-attribute-string"/>
        <xsl:with-param name="new-delta-attribute-string" select="$new-delta-attribute-string"/>
        <xsl:with-param name="attribute-names" select="$other-attribute-names"/>
      </xsl:call-template>
    </xsl:if>
  </xsl:template>
  
  <!-- Returns a string denoting the value of attribute-name from delta-attribute-string -->
  <xsl:template name="find-attribute-value">
    <xsl:param name="delta-attribute-string"/>
    <xsl:param name="attribute-name"/>
    <!-- find the name of the first attribute -->
    <xsl:variable name="first-attribute-name"
      select="substring-before($delta-attribute-string,'=')"/>
    <xsl:if test="string-length($first-attribute-name)>0">
      <xsl:variable name="attribute-value-delimiter"
        select="substring(substring-after($delta-attribute-string,'='),1,1)"/>
      <xsl:variable name="attribute-value-and-rest"
        select="substring-after($delta-attribute-string, $attribute-value-delimiter)"/>
      <xsl:choose>
        <xsl:when test="$attribute-name=$first-attribute-name">
          <!-- then we get the value and return it -->
          <xsl:value-of
            select="substring-before(
            $attribute-value-and-rest,$attribute-value-delimiter)"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="find-attribute-value">
            <xsl:with-param name="delta-attribute-string"
              select="substring(substring-after(
              $attribute-value-and-rest, $attribute-value-delimiter),2)"/>
            <xsl:with-param name="attribute-name" select="$attribute-name"/>
          </xsl:call-template>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:if>
  </xsl:template>
  
  <!-- Templates for finding which attributes are added, deleted and changed -->
  
  <xsl:template name="nmtokens-set-difference">
    <xsl:param name="first-string"/>
    <xsl:param name="second-string"/>
    <xsl:call-template name="nmtokens-set-function">
      <xsl:with-param name="first-string"
        select="concat(normalize-space($first-string), ' ')"/>
      <xsl:with-param name="second-string" select="$second-string"/>
      <xsl:with-param name="function" select="string('difference')"/>
    </xsl:call-template>
  </xsl:template>
  
  <xsl:template name="nmtokens-set-intersection">
    <xsl:param name="first-string"/>
    <xsl:param name="second-string"/>
    <xsl:call-template name="nmtokens-set-function">
      <xsl:with-param name="first-string"
        select="concat(normalize-space($first-string), ' ')"/>
      <xsl:with-param name="second-string" select="$second-string"/>
      <xsl:with-param name="function" select="string('intersection')"/>
    </xsl:call-template>
  </xsl:template>
  
  <!-- Returns a string containing the values of NMTOKENS in first-string but not in second-string
  if function='difference' or values in both strings if function='intersection'
  Note that there must be no space before the first token and there must be one space at the end,
  i.e. after the last token. This is for efficiency to save repeated normalization-->
  <xsl:template name="nmtokens-set-function">
    <xsl:param name="first-string"/>
    <xsl:param name="second-string"/>
    <xsl:param name="function"/>
    <!-- find the name of the first token -->
    <xsl:variable name="first-token"
      select="concat(substring-before($first-string,' '), ' ')"/>
    <!-- space is needed above so that we do not find nmtokens that are substrings of other tokens -->
    <xsl:if test="string-length($first-token)>1">
      <xsl:if test="($function='intersection' and contains($second-string, $first-token))
        or
        ($function='difference' and not(contains($second-string, $first-token)))">
        <xsl:value-of select="$first-token"/>
      </xsl:if>
      <xsl:variable name="the-rest"
        select="substring-after($first-string, ' ')"/>
      <xsl:if test="string-length($the-rest)>1">
        <xsl:call-template name="nmtokens-set-function">
          <xsl:with-param name="first-string" select="$the-rest"/>
          <xsl:with-param name="second-string" select="$second-string"/>
          <xsl:with-param name="function" select="$function"/>
        </xsl:call-template>
      </xsl:if>
    </xsl:if>
  </xsl:template>
  
</xsl:stylesheet>