// $Id: WordByWordOutfilter2.java 4699 2008-10-16 14:19:53Z nigelw $
// (c) 2005 DeltaXML Ltd. All rights reserved.

package com.deltaxml.pipe.filters;

import com.deltaxml.pipe.XMLFilterImpl2;
import org.xml.sax.helpers.AttributesImpl;
import org.xml.sax.SAXException;
import org.xml.sax.Attributes;

/**
 * <p>Merges consecutive modification notification elements (<code>deltaxml:PCDATAmodify</code>).</p>
 * <p>The XML comparator outputs PCDATA modifications as follows.
 * <pre>
 * &lt;deltaxml:PCDATAmodify&gt;
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;deltaxml:PCDATAold&gt;
 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Old data here
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;/deltaxml:PCDATAold&gt;
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;deltaxml:PCDATAnew&gt;
 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;New data here
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;/deltaxml:PCDATAnew&gt;
 * &lt;/deltaxml:PCDATAmodify&gt;
 * </pre>
 * If XML files are compared using a detailed comparison, in other words their PCDATA is compared word by word,
 * the output can potentially have consecutive <code>deltaxml:PCDATAmodify</code> elements within it due to
 * consecutive words being modified.</p>
 * <p>This filter concatenates consecutive <code>deltaxml:PCDATAmodify</code> elements so that consecutive
 * modifications are represented by a single <code>deltaxml:PCDATAmodify</code> element.</p>
 * Thus,
 * <pre>
 * &lt;deltaxml:PCDATAmodify&gt;
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;deltaxml:PCDATAold&gt;
 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Old
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;/deltaxml:PCDATAold&gt;
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;deltaxml:PCDATAnew&gt;
 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;New
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;/deltaxml:PCDATAnew&gt;
 * &lt;/deltaxml:PCDATAmodify&gt;
 * 
 * &lt;deltaxml:PCDATAmodify&gt;
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;deltaxml:PCDATAold&gt;
 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;text
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;/deltaxml:PCDATAold&gt;
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;deltaxml:PCDATAnew&gt;
 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;data
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;/deltaxml:PCDATAnew&gt;
 * &lt;/deltaxml:PCDATAmodify&gt;
 * </pre>
 * would be converted to
 * <pre>
 * &lt;deltaxml:PCDATAmodify&gt;
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;deltaxml:PCDATAold&gt;
 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Old text
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;/deltaxml:PCDATAold&gt;
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;deltaxml:PCDATAnew&gt;
 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;New data
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;/deltaxml:PCDATAnew&gt;
 * &lt;/deltaxml:PCDATAmodify&gt;
 * </pre>
 * by this filter.
 * 
 * <p><code>WordByWordOutfilter3</code> should always be used in conjunction with
 * {@link WordByWordInfilter} and {@link WordByWordOutfilter1}. It is designed to be used
 * as a post-filter and should be placed immediately after <code>WordByWordOutfilter1</code>
 * in the <code>XMLComparator</code> output pipeline.</p>
 * 
 * <p><strong>Note: </strong>This class has not been designed to be extended,
 * therefore to err on the side of caution, it has been declared final.</p>
 * 
 * @version $Id: WordByWordOutfilter2.java 4699 2008-10-16 14:19:53Z nigelw $
 * @author Tristan Mitchell.
 * @see com.deltaxml.pipe.filters.WordByWordInfilter
 * @see com.deltaxml.pipe.filters.WordByWordOutfilter1
 */

public final class WordByWordOutfilter2 extends XMLFilterImpl2
{
    static String DXML_PREFIX= "deltaxml";
    static String PCDATA_MODIFY_LOCAL= "PCDATAmodify";
    static String PCDATA_MODIFY= DXML_PREFIX + ":" + PCDATA_MODIFY_LOCAL;
    static String PCDATA_OLD_LOCAL= "PCDATAold";
    static String PCDATA_OLD= DXML_PREFIX + ":" + PCDATA_OLD_LOCAL;
    static String PCDATA_NEW_LOCAL= "PCDATAnew";
    static String PCDATA_NEW= DXML_PREFIX + ":" + PCDATA_NEW_LOCAL;
    static String DELTAXML_NS= "http://www.deltaxml.com/ns/well-formed-delta-v1";
    boolean isPcdataOld= false; //true if current element is a PCDATAold element
    boolean isPcdataNew= false; //true if current element is a PCDATAnew element
    boolean pcdataStringsEmpty= true; //false if either of the following Strings contain text
    String pcdataOld= new String();
    String pcdataNew= new String();

    String characters= "";
    boolean tempStringEmpty= true;
    /**
     * tempString is used to store spaces between PCDATAmodify elements that relate to spaces in the text
     */
    String tempString= new String();
    
    /**
     * Creates a new instance of <code>WordByWordOutfilter3</code>.
     */
    public WordByWordOutfilter2()
    {
        super();
    }
    
    /**
     * Overrides the default <code>startPrefixMapping</code> method.
     * @throws SAXException the superclass may throw an exception during processing.
     * @see XMLFilterImpl#startPrefixMapping(String, String)
     */
    public void startPrefixMapping(String prefix, String uri)
            throws SAXException
    {
        if(characters.length() > 0) {
            outputCharacters();
        }
        super.startPrefixMapping(prefix, uri);
    }
    
    /**
     * Overrides the default <code>startElement</code> method.
     * @throws SAXException the superclass may throw an exception during processing.
     * @see XMLFilterImpl#startElement(String, String, String, Attributes)
     */
    public void startElement(String uri, String localName, String qName, Attributes atts)
        throws SAXException
    {
        /**
         * startElement performs the following:
         * 1) if the element is a PCDATAmodify element and the tempString is not empty, one of
         * two things can occur a)if the tempString consists of a space character this is concatenated
         * onto both the pcdataOld and pcdataNew strings b) if the tempString is anything other than a space
         * character, the previous PCDATAmodify block is output followed by the tempString as a character event
         * 2) if the element is a PCDATAold or PCDATAnew element, the relevant flags are set
         * 3) otherwise output any temporarily stored strings as a PCDATAmodify block
         */
        if(characters.length() > 0) {
            outputCharacters();
        }
        if(qName.equals(PCDATA_MODIFY)) {
            
            if(!tempStringEmpty) {
                if(tempString.equals(" ")) {
                    pcdataOld= pcdataOld.concat(tempString);
                    pcdataNew= pcdataNew.concat(tempString);
                    tempString= "";
                    tempStringEmpty= true;
                } else {
                    outputStrings();
                    super.characters(tempString.toCharArray(), 0, tempString.length());
                    tempString= "";
                    tempStringEmpty= true;
                }
            }
            pcdataStringsEmpty= false;
        } else if(qName.equals(PCDATA_OLD)) {
            isPcdataOld= true;
        } else if(qName.equals(PCDATA_NEW))	{
            isPcdataNew= true;
        } else {
            if(!pcdataStringsEmpty) {
                outputStrings();
            }
            if(!tempStringEmpty) {
                super.characters(tempString.toCharArray(), 0, tempString.length());
                tempString= "";
                tempStringEmpty= true;
            }
            AttributesImpl atts1= new AttributesImpl(atts);
            for (int i= atts.getLength()-1; i >= 0; i--) {
                String name= atts1.getQName(i);
                if(name.startsWith("xmlns:")) {
                    atts1.removeAttribute(i);
                }
            }
            super.startElement(uri, localName, qName, atts1);
        }
    }

    /**
     * Overrides the default <code>endElement</code> method.
     * @throws SAXException the superclass may throw an exception during processing.
     * @see XMLFilterImpl#endElement(String, String, String)
     */
    public void endElement(String uri, String localName, String qName)
        throws SAXException
    {
        /**
         * endElement performs the following:
         * 1) resets flags for PCDATAold and PCDATAnew
         * 2) otherwise, if the element is not a PCDATAmodify element, output anything
         * held in the temporary strings then output the end element
         */
        if(characters.length() > 0) {
            outputCharacters();
        }
        if(qName.equals(PCDATA_OLD)) {
            isPcdataOld= false;
        } else if(qName.equals(PCDATA_NEW)) {
            isPcdataNew= false;
        } else if(!(qName.equals(PCDATA_MODIFY))) {
            //if the name is something other than PCDATA_OLD _NEW or _MODIFY we potentially need
            //to output the strings.
            if(!pcdataStringsEmpty) {
                outputStrings();
            }
            if(!tempStringEmpty) {
                super.characters(tempString.toCharArray(), 0, tempString.length());
                tempString= "";
                tempStringEmpty= true;
            }
            super.endElement(uri, localName, qName);
        }
    }

    /**
     * Overrides the default <code>characters</code> method.
     * @throws SAXException the superclass may throw an exception during processing.
     * @see XMLFilterImpl#characters(char[], int, int)
     */
    public void characters(char[] ch, int start, int length)
        throws SAXException
    {
        /**
         * characters performs the following:
         * 1) If in PCDATAold, concatenate the characters onto the PCDATAold temp string
         * 2) If in PCDATAnew, concatenate the characters onto the PCDATAnew temp string
         * 3) otherwise, if the tempStrings haven't been output, add any characters to the
         * inter-element tempString
         * 4) otherwise, store characters in the String 'characters' as they are received
         */
        if(isPcdataOld)	{
            pcdataOld= pcdataOld.concat(String.copyValueOf(ch, start, length));
        } else if(isPcdataNew) {
            pcdataNew= pcdataNew.concat(String.copyValueOf(ch, start, length));
        } else if(!pcdataStringsEmpty) {
            tempString= tempString.concat(String.copyValueOf(ch, start, length));
            tempStringEmpty= false;
        } else {
            characters= characters + new String(ch, start, length);
        }
    }

    /**
     * outputStrings is used to create a new PCDATAmodify block that contains the concatenation	
     * of all the original consecutive PCDATAmodify blocks in the format
     * <PCDATAmodify>
     * 		<PCDATAold>	
     *			old text
     *		</PCDATAold>
     * 		<PCDATAnew>
     *			new text
     * 		</PCDATAnew>
     * </PCDATAmodify>
     */
    private void outputStrings()
        throws SAXException
    {
        if(tempString.equals(" ")) {
            pcdataOld= pcdataOld.concat(tempString);
            pcdataNew= pcdataNew.concat(tempString);
            tempString ="";
            tempStringEmpty= true;
        }
        super.startElement(DELTAXML_NS, PCDATA_MODIFY_LOCAL, PCDATA_MODIFY, new AttributesImpl());
        super.startElement(DELTAXML_NS, PCDATA_OLD_LOCAL, PCDATA_OLD, new AttributesImpl());
        if(pcdataOld.length() != 0) {
            super.characters(pcdataOld.toCharArray(), 0, pcdataOld.length());
        }
        super.endElement(DELTAXML_NS, PCDATA_OLD_LOCAL, PCDATA_OLD);
        super.startElement(DELTAXML_NS, PCDATA_NEW_LOCAL, PCDATA_NEW, new AttributesImpl());
        if(pcdataNew.length() != 0) {
            super.characters(pcdataNew.toCharArray(), 0, pcdataNew.length());
        }
        super.endElement(DELTAXML_NS, PCDATA_NEW_LOCAL, PCDATA_NEW);
        super.endElement(DELTAXML_NS, PCDATA_MODIFY_LOCAL, PCDATA_MODIFY);
        pcdataOld= "";
        pcdataNew= "";
        pcdataStringsEmpty= true;
    }
    
    /**
     * This method is included to ensure characters are output in the same
     * SAXEvent style as an equivalent xsl filter - in this case all consecutive characters
     * must be output in one SAXEvent
     */
    private void outputCharacters() throws SAXException
    {
        char[] chars= characters.toCharArray();
        super.characters(chars, 0, chars.length);
        characters= "";
    }
}