// $Id: WordByWordOutfilter1.java 4699 2008-10-16 14:19:53Z nigelw $
// Copyright (c) 2006 DeltaXML Ltd. All rights reserved

package com.deltaxml.pipe.filters;

import java.util.HashMap;
import java.util.Map;

import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.AttributesImpl;
import com.deltaxml.pipe.XMLFilterImpl2;

/**
 * <p>Concatenates contents of consecutive <code>deltaxml:word</code> and <code>deltaxml:space</code>
 * elements that have the same attributes. Attributes on the elements (representing formatting elements)
 * are output as elements around the pcdata inside the word and space elements.</p>
 * 
 * <p><strong>Example</strong></p>
 * Comparing (after filtering with <code>WordByWordInfilter</code>)
 * <pre>
 * &lt;p&gt;sample of formatting changes&lt;/p&gt;
 * </pre>
 * with
 * <pre>
 * &lt;p&gt;sample of &lt;b deltaxml:format="true"&gt;formatting&lt;/b&gt; changes&lt;/p&gt;
 * </pre>
 * gives the delta file (pretty printed)
 * <pre>
 * &lt;p deltaxml:delta="WFmodify"&gt;
 * &nbsp;&nbsp;&lt;deltaxml:word deltaxml:delta="unchanged"&gt;sample&lt;/deltaxml:word&gt;
 * &nbsp;&nbsp;&lt;deltaxml:space deltaxml:delta="unchanged"&gt; &lt;/deltaxml:space&gt;
 * &nbsp;&nbsp;&lt;deltaxml:word deltaxml:delta="unchanged"&gt;of&lt;/deltaxml:word&gt;
 * &nbsp;&nbsp;&lt;deltaxml:space deltaxml:delta="unchanged"&gt; &lt;/deltaxml:space&gt;
 * &nbsp;&nbsp;&lt;deltaxml:word deltaxml:delta="WFmodify" deltaxml:new-attributes="b=''"&gt;formatting&lt;/deltaxml:word&gt;
 * &nbsp;&nbsp;&lt;deltaxml:space deltaxml:delta="unchanged"&gt; &lt;/deltaxml:space&gt;
 * &nbsp;&nbsp;&lt;deltaxml:word deltaxml:delta="unchanged"&gt;changes&lt;/deltaxml:word&gt;
 * &lt;/p&gt;
 * </pre>
 * 
 * <p>This filter will process the delta file to remove the word and space elements and show the
 * changes to their contents as follows (pretty printed):</p>
 * 
 * <pre>
 * &lt;p deltaxml:delta="WFmodify"&gt;
 * &nbsp;&nbsp;sample of 
 * &nbsp;&nbsp;&lt;deltaxml:exchange&gt;
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;deltaxml:old&gt;formatting&lt;/deltaxml:old&gt;
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;deltaxml:new&gt;
 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&lt;b deltaxml:delta="add"&gt;formatting&lt;/b&gt;
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;/deltaxml:new&gt;
 * &nbsp;&nbsp;&lt;/deltaxml:exchange&gt;
 * &nbsp;&nbsp;changes
 * &lt;/p&gt;
 * </pre>
 * 
 * <p><code>WordByWordOutfilter1</code> should always be used in conjunction with
 * {@link WordByWordInfilter}. It is designed to be used as a post-filter and should be
 * placed immediately after the <code>XMLComparator</code> in the pipeline.</p>
 * 
 * <p><strong>Note: </strong>This class has not been designed to be extended,
 * therefore to err on the side of caution, it has been declared final.</p>
 * 
 * @version $Id: WordByWordOutfilter1.java 4699 2008-10-16 14:19:53Z nigelw $
 * @author Tristan Mitchell
 * @see WordByWordInfilter
 */
public final class WordByWordOutfilter1 extends XMLFilterImpl2
{
  private final String DELTAXML_NS= "http://www.deltaxml.com/ns/well-formed-delta-v1";
  private final String DELTAXML_PREFIX= "deltaxml";
  private final String SPACE_LOCAL_NAME= "space";
  private final String WORD_LOCAL_NAME= "word";
  private final String PUNCTUATION_LOCAL_NAME= "punctuation";
  private final String DELTA_LOCAL_NAME= "delta";
  private final String PCDATAMODIFY_LOCAL_NAME= "PCDATAmodify";
  private final String PCDATAOLD_LOCAL_NAME= "PCDATAold";
  private final String PCDATANEW_LOCAL_NAME= "PCDATAnew";
  private final String FORMAT_LOCAL_NAME= "format";
  private final String OLD_LOCAL_NAME= "old";
  private final String NEW_LOCAL_NAME= "new"; 
  private final String EXCHANGE_LOCAL_NAME= "exchange";
  private final String OLD_ATTS_LOCAL_NAME= "old-attributes";
  private final String NEW_ATTS_LOCAL_NAME= "new-attributes";
  private final String UNCHANGED= "unchanged";
  private final String ADDED= "add";
  private final String DELETED= "delete";
  private final String MODIFY= "WFmodify";
 
  //this Set is used to store the local names of deltaxml elements we are concerned about in this filter
  //all other 
//  private Set deltaXMLElementsToProcess= new HashSet();
  
  private boolean inExchange= false;
  private boolean inPCDataOld= false;
  private boolean inPCDataNew= false;
  private boolean inWordPunctuationOrSpace= false;
  private boolean lastElemWasModifiedWord= false;
  private boolean inSpecialSpaceCase= false;
  
  private StringBuffer oldChars= new StringBuffer();
  private StringBuffer newChars= new StringBuffer();
  private StringBuffer unchangedChars= new StringBuffer();
  
  private String currentDeltaValue= null; ///this is the latest received delta attribute
  
  //it is possible to store prefix to namespace mappings in a Map since the comparator produces
  //unique prefixes and this is an output filter
  private Map namespaces= new HashMap();
  
  private AttributesImpl currentAtts= null; //this will only ever be the atts off a word or space
  
  /**
   * Overrides the default <code>startDocument</code> method. This version of the method performs internal operations.
   * @throws SAXException the superclass may throw an exception during processing.
   * @see XMLFilterImpl#startDocument()
   */
  public void startDocument() throws SAXException {
    // add the xml namespace to our namespaces map as this is available without being declared
    namespaces.put("xml", "http://www.w3.org/XML/1998/namespace");
    super.startDocument();
  }

  /**
   * Overrides the default <code>startElement</code> method. This version of the method determines whether to
   * output the element as it is, or store it for outputting at a later stage.
   * @throws SAXException the superclass may throw an exception during processing.
   * @see XMLFilterImpl#startElement(String, String, String, Attributes)
   */
  public void startElement(String uri, String localName, String qName, Attributes atts) throws SAXException {
    boolean outputThisElement= false;
    
    if (uri.equals(DELTAXML_NS)) {
      //in a deltaxml element, could be word, space, exchange or PCDATAmodify (which will ALWAYS be in a word or space)
      
      //we need to cater for one sepcial case of space:
      //if an unchanged space follows a modified word with the same attributes (apart from the delta att), we need to
      //add the space to the modified word.
      if(lastElemWasModifiedWord && localName.equals(SPACE_LOCAL_NAME) && UNCHANGED.equals(atts.getValue(DELTAXML_NS, DELTA_LOCAL_NAME)) && attsEqual(atts, true)) {
        //we are in our special case word, this could be an unchaged space in between 2 modified words,
        //in which case we want to include it in the modified text
        outputThisElement= false;
        inSpecialSpaceCase= true;
//        System.out.println("IN SPECIAL SPACE CASE");
      } else if(localName.equals(WORD_LOCAL_NAME) || localName.equals(SPACE_LOCAL_NAME) || localName.equals(PUNCTUATION_LOCAL_NAME)) {
        // in word, punctuation or space (treat them the same)
        inWordPunctuationOrSpace= true;
        inSpecialSpaceCase= false;
        outputThisElement= false;
        if(!attsEqual(atts, false)) {
          output();
          currentAtts= new AttributesImpl(atts);
        }
        if(atts.getValue(DELTAXML_NS, DELTA_LOCAL_NAME) != null) {
          currentDeltaValue= atts.getValue(DELTAXML_NS, DELTA_LOCAL_NAME);
        }
        lastElemWasModifiedWord= (MODIFY.equals(currentDeltaValue) && localName.equals(WORD_LOCAL_NAME));
      } else if(localName.equals(EXCHANGE_LOCAL_NAME)) {
        ///in exchange outer element
        inSpecialSpaceCase= false;
        lastElemWasModifiedWord= false;
        output();
        inExchange= true;
        outputThisElement= true;
      } else if(localName.equals(OLD_LOCAL_NAME)) {
        //in exchange old element
        inSpecialSpaceCase= false;
        lastElemWasModifiedWord= false;
        currentDeltaValue= DELETED;
        outputThisElement= true;
      } else if(localName.equals(NEW_LOCAL_NAME)) {
        //in exchange new element
        inSpecialSpaceCase= false;
        lastElemWasModifiedWord= false;
        currentDeltaValue= ADDED;
        outputThisElement= true;
      } else if(localName.equals(PCDATAMODIFY_LOCAL_NAME)) {
        //in PCDATAmodify outer element
        //if we are NOT in a deltaxml:word or deltaxml:space we want to output this element
        outputThisElement= !inWordPunctuationOrSpace;
        inSpecialSpaceCase= false;
      } else if(localName.equals(PCDATAOLD_LOCAL_NAME)) {
        //in PCDATAold element
        //if we are NOT in a deltaxml:word or deltaxml:space we want to output this element
        outputThisElement= !inWordPunctuationOrSpace;
        inSpecialSpaceCase= false;
        inPCDataOld= true;
      } else if(localName.equals(PCDATANEW_LOCAL_NAME)) {
        //in PCDATAnew element
        //if we are NOT in a deltaxml:word or deltaxml:space we want to output this element
        outputThisElement= !inWordPunctuationOrSpace;
        inSpecialSpaceCase= false;
        inPCDataNew= true;
      } else {
        inSpecialSpaceCase= false;
        lastElemWasModifiedWord= false;
        output();
        outputThisElement= true;
      }
    } else {
      //in other type of element - if it has a delta value, we want to store it. Output this element whatever (I think)
      lastElemWasModifiedWord= false;
      inSpecialSpaceCase= false;
      output();  //I THINK
      outputThisElement= true;
      if(atts.getValue(DELTAXML_NS, DELTA_LOCAL_NAME) != null) {
        //we have a delta value
        currentDeltaValue= atts.getValue(DELTAXML_NS, DELTA_LOCAL_NAME);
      }
    }
    
    if(outputThisElement) {
      super.startElement(uri, localName, qName, atts);
    }
  }
  
  /**
   * Overrides the default <code>endElement</code> method. This version of the method determines whether to
   * output the element as it is, or store it for outputting at a later stage.
   * @throws SAXException the superclass may throw an exception during processing.
   * @see XMLFilterImpl#endElement(String, String, String)
   */
  public void endElement(String uri, String localName, String qName) throws SAXException {
    
    boolean outputThisElement= false; 
    if(uri.equals(DELTAXML_NS)) {
      //in a deltaxml element
      if(localName.equals(WORD_LOCAL_NAME) || localName.equals(SPACE_LOCAL_NAME) || localName.equals(PUNCTUATION_LOCAL_NAME)) {
        inWordPunctuationOrSpace= false;
        outputThisElement= false;
      } else if(localName.equals(EXCHANGE_LOCAL_NAME)) {
        outputThisElement= true;
        inExchange= false;
      } else if(localName.equals(OLD_LOCAL_NAME)) {
        outputThisElement= true;
        output(); // output any words that were in this element
      } else if(localName.equals(NEW_LOCAL_NAME)) {
        outputThisElement= true;
        output(); //output any words that were in this element
      } else if(localName.equals(PCDATAMODIFY_LOCAL_NAME)) {
        //if we are NOT in a deltaxml:word or deltaxml:space we want to output this element
        outputThisElement= !inWordPunctuationOrSpace;
      } else if(localName.equals(PCDATAOLD_LOCAL_NAME)) {
        //if we are NOT in a deltaxml:word or deltaxml:space we want to output this element
        outputThisElement= !inWordPunctuationOrSpace;
        inPCDataOld= false;
      } else if(localName.equals(PCDATANEW_LOCAL_NAME)) {
        //if we are NOT in a deltaxml:word or deltaxml:space we want to output this element
        outputThisElement= !inWordPunctuationOrSpace;
        inPCDataNew= false;
      } else {
        outputThisElement= true;
      }
    } else {
      //in another kind of element
      output(); //output any words that were in this element
      outputThisElement= true;
      lastElemWasModifiedWord= false;
    }
    
    if(outputThisElement) {
      super.endElement(uri, localName, qName);
    } 
  }
  
  /**
   * Overrides the default <code>startPrefixMapping<code> method. This version of the method performs internal operations.
   * @throws SAXException the superclass may throw an exception during processing.
   * @see XMLFilterImpl#startPrefixMapping(String, String)
   */
  public void startPrefixMapping(String prefix, String uri) throws SAXException {
    output();
    namespaces.put(prefix, uri);
    super.startPrefixMapping(prefix, uri);
  }
  
  /**
   * Overrides the default <code>endPrefixMapping</code> method. This version of the method performs internal operations.
   * @throws SAXException the superclass may throw an exception during processing.
   * @see XMLFilterImpl#endPrefixMapping(String)
   */
  public void endPrefixMapping(String prefix) throws SAXException {
    namespaces.remove(prefix);
    super.endPrefixMapping(prefix);
  }
  
  /**
   * Overrides the default <code>characters</code> method. This version of the method stores characters to be processed later.
   * @throws SAXException the superclass may throw an exception during processing.
   * @see XMLFilterImpl#characters(char[], int, int)
   */
  public void characters(char [] ch, int start, int length) throws SAXException {
    //only store the characters if we are inside a deltaxml:word or deltaxml:space element.
    //if we aren't, we must be in something that is NOT being compared WordByWord
    if (inSpecialSpaceCase) {
      newChars.append(ch, start, length);
      oldChars.append(ch, start, length);
    } else if(inWordPunctuationOrSpace) {
      if (currentDeltaValue.equals(UNCHANGED)) {
        unchangedChars.append(ch, start, length);
      } else if(currentDeltaValue.equals(ADDED)) {
        newChars.append(ch, start, length);
      } else if(currentDeltaValue.equals(DELETED)) {
        oldChars.append(ch, start, length);
      } else if(currentDeltaValue.equals(MODIFY)) {
        if(inPCDataOld) {
          oldChars.append(ch, start, length);
        } else if(inPCDataNew) {
          newChars.append(ch, start, length);
        } else {
          oldChars.append(ch, start, length);
          newChars.append(ch, start, length);
        }
      } 
    } else {
      super.characters(ch, start, length);
    }
  }
  
  private void outputCharacters(StringBuffer stored) throws SAXException {
    char[] chars= new char[stored.length()];
    stored.getChars(0, stored.length(), chars, 0);
    super.characters(chars, 0, chars.length);
  }
  
  private void outputStartTags(Attributes atts, boolean deltaAtt) throws SAXException {
    //modify delta attribute needs to go on all start tags - other just go on the first one
    if(currentDeltaValue.equals(MODIFY)) {
      for(int i= 0; i < atts.getLength(); i++) {
        AttributesImpl attributes= decodeAttString(atts.getValue(i));
        if(deltaAtt) {
          attributes.addAttribute(DELTAXML_NS, DELTA_LOCAL_NAME, DELTAXML_PREFIX + ":" + DELTA_LOCAL_NAME, "CDATA", currentDeltaValue);
        }
        super.startElement(atts.getURI(i), atts.getLocalName(i), atts.getQName(i), attributes);
      }
    } else {
      AttributesImpl attributes= decodeAttString(atts.getValue(0));
      if(deltaAtt) {
        attributes.addAttribute(DELTAXML_NS, DELTA_LOCAL_NAME, DELTAXML_PREFIX + ":" + DELTA_LOCAL_NAME, "CDATA", currentDeltaValue);
      }
      super.startElement(atts.getURI(0), atts.getLocalName(0), atts.getQName(0), attributes);
      for(int i= 1; i < atts.getLength(); i++) {
        super.startElement(atts.getURI(i), atts.getLocalName(i), atts.getQName(i), decodeAttString(atts.getValue(i)));
      }
    }
  }
  
  private AttributesImpl decodeAttString(String attString) {
    AttributesImpl atts= new AttributesImpl();
    //always include a deltaxml:format="true" attribute
    atts.addAttribute(DELTAXML_NS, FORMAT_LOCAL_NAME, DELTAXML_PREFIX + ":" + FORMAT_LOCAL_NAME, "CDATA", "true");
    
    int start= 0;
    int index;
    while ((index= attString.indexOf('=', start)) != -1) {
      String qName= "";
      String val= "";
      qName= attString.substring(start, index);
      char delimiter= attString.charAt(index+1);
      int end= attString.indexOf(delimiter, index+2);
      val= attString.substring(index+2, end);
      //add the Attribute to atts
      String uri= "";
      String localName= qName;
      if (qName.indexOf(":") != -1) {
        //we have a prefixed name
        String prefix= qName.substring(0, qName.indexOf(":"));
        localName= qName.substring(qName.indexOf(":")+1);
        uri= (String)namespaces.get(prefix);
      }
      atts.addAttribute(uri, localName, qName, "CDATA", val);
      start= end+2;
    }
    return atts;
  }
  
  private void outputEndTags(Attributes atts) throws SAXException {
    for (int i= atts.getLength()-1; i >= 0; i--) {
      super.endElement(atts.getURI(i), atts.getLocalName(i), atts.getQName(i));
    }
  }
  
  private void outputPCDataModify() throws SAXException {
    super.startElement(DELTAXML_NS, PCDATAMODIFY_LOCAL_NAME, DELTAXML_PREFIX + ":" + PCDATAMODIFY_LOCAL_NAME, new AttributesImpl());
    super.startElement(DELTAXML_NS, PCDATAOLD_LOCAL_NAME, DELTAXML_PREFIX + ":" + PCDATAOLD_LOCAL_NAME, new AttributesImpl());
    outputCharacters(oldChars);
    super.endElement(DELTAXML_NS, PCDATAOLD_LOCAL_NAME, DELTAXML_PREFIX + ":" + PCDATAOLD_LOCAL_NAME);
    super.startElement(DELTAXML_NS, PCDATANEW_LOCAL_NAME, DELTAXML_PREFIX + ":" + PCDATANEW_LOCAL_NAME, new AttributesImpl());
    outputCharacters(newChars);
    super.endElement(DELTAXML_NS, PCDATANEW_LOCAL_NAME, DELTAXML_PREFIX + ":" + PCDATANEW_LOCAL_NAME);
    super.endElement(DELTAXML_NS, PCDATAMODIFY_LOCAL_NAME, DELTAXML_PREFIX + ":" + PCDATAMODIFY_LOCAL_NAME);
  }
  
  private void outputExchange() throws SAXException {
    AttributesImpl oldFormatTags= new AttributesImpl();
    AttributesImpl newFormatTags= new AttributesImpl();
    
    if(currentAtts.getValue(DELTAXML_NS, OLD_ATTS_LOCAL_NAME) != null) {
      oldFormatTags= attStringToAttributes(currentAtts.getValue(DELTAXML_NS, OLD_ATTS_LOCAL_NAME));
      //now remove the att so we can see if we have any unchanged ones
      currentAtts.removeAttribute(currentAtts.getIndex(DELTAXML_NS, OLD_ATTS_LOCAL_NAME));
    }
    if(currentAtts.getValue(DELTAXML_NS, NEW_ATTS_LOCAL_NAME) != null) {
      newFormatTags= attStringToAttributes(currentAtts.getValue(DELTAXML_NS, NEW_ATTS_LOCAL_NAME));
      //now remove the att so we can see if we have any unchanged ones
      currentAtts.removeAttribute(currentAtts.getIndex(DELTAXML_NS, NEW_ATTS_LOCAL_NAME));
    }
    //remove the delta att if it is present(it should be)
    int deltaIndex= currentAtts.getIndex(DELTAXML_NS, DELTA_LOCAL_NAME);
    if (deltaIndex != -1) {
      currentAtts.removeAttribute(deltaIndex);
    }
    //output any unchange formatting tags with moddify delta att
    if(currentAtts.getLength() > 0) {
      currentDeltaValue= MODIFY;
      outputStartTags(currentAtts, true);
    }
    super.startElement(DELTAXML_NS, EXCHANGE_LOCAL_NAME, DELTAXML_PREFIX + ":" + EXCHANGE_LOCAL_NAME, new AttributesImpl());
    super.startElement(DELTAXML_NS, OLD_LOCAL_NAME, DELTAXML_PREFIX + ":" + OLD_LOCAL_NAME, new AttributesImpl());
    if (oldFormatTags.getLength() > 0) {
      currentDeltaValue= DELETED;
      outputStartTags(oldFormatTags, true);
    }
    outputCharacters(oldChars);
    if (oldFormatTags.getLength() > 0) {
      outputEndTags(oldFormatTags);
    }
    super.endElement(DELTAXML_NS, OLD_LOCAL_NAME, DELTAXML_PREFIX + ":" + OLD_LOCAL_NAME);
    super.startElement(DELTAXML_NS, NEW_LOCAL_NAME, DELTAXML_PREFIX + ":" + NEW_LOCAL_NAME, new AttributesImpl());
    if (newFormatTags.getLength() > 0) {
      currentDeltaValue= ADDED;
      outputStartTags(newFormatTags, true);
    }
    outputCharacters(newChars);
    if (newFormatTags.getLength() > 0) {
      outputEndTags(newFormatTags);
    }
    super.endElement(DELTAXML_NS, NEW_LOCAL_NAME, DELTAXML_PREFIX + ":" + NEW_LOCAL_NAME);
    super.endElement(DELTAXML_NS, EXCHANGE_LOCAL_NAME, DELTAXML_PREFIX + ":" + EXCHANGE_LOCAL_NAME);
    //output any unchaged formatting end tags
    if (currentAtts.getLength() > 0) {
      outputEndTags(currentAtts);
    }
  }
  
  private void output() throws SAXException {
    //only do something if we have any characters saved
    if(unchangedChars.length() > 0 || newChars.length() > 0 || oldChars.length() > 0) {
      if(currentDeltaValue.equals(UNCHANGED)) {
        //if we have atts, output them as format tags, otherwise just output the characters
        if(currentAtts != null) {
          int deltaIndex= currentAtts.getIndex(DELTAXML_NS, DELTA_LOCAL_NAME);
          if (deltaIndex != -1) {
            currentAtts.removeAttribute(deltaIndex);
          }
          //see if we have any atts left
          if (currentAtts.getLength() > 0) {
            //output the start format tags
            outputStartTags(currentAtts, deltaIndex != -1); //if we removed a deltaIndex, we need to add it on to the first formatting tag
            //output the characters
            outputCharacters(unchangedChars);
            //output the end format tags
            outputEndTags(currentAtts);
          } else {
            outputCharacters(unchangedChars);
          }
        } else {
          outputCharacters(unchangedChars);
        }
      } else if(currentDeltaValue.equals(ADDED) || currentDeltaValue.equals(DELETED)) {
        // if there are format tags, output an added/deleted element, otherwise output a PCDATAmodify
        if (currentAtts != null) {
          if(currentAtts.getLength() != 0) {
            int deltaIndex= currentAtts.getIndex(DELTAXML_NS, DELTA_LOCAL_NAME);
            if (deltaIndex != -1) {
              currentAtts.removeAttribute(deltaIndex);
            }
            //see if we have any atts left
            if(currentAtts.getLength() > 0) {
              outputStartTags(currentAtts, deltaIndex != -1);
              outputCharacters(currentDeltaValue.equals(ADDED) ? newChars : oldChars);
              outputEndTags(currentAtts);
            } else {
              if (inExchange) {
                outputCharacters(currentDeltaValue.equals(ADDED) ? newChars : oldChars);
              } else {
                outputPCDataModify();
              }
            }
          } else {
            //no atts to start with - we must be inside an added or deleted element - lets output characters
            outputCharacters(currentDeltaValue.equals(ADDED) ? newChars : oldChars);
          }
        } else {
          outputPCDataModify();
        }
      } else if(currentDeltaValue.equals(MODIFY)) {
        //3 scenarios 1) format change only, 2) PCDATA change only, 3) both changes
        // 1) and 3) require an exchange element, 2) requires PCDATAmodify only
        if (currentAtts.getValue(DELTAXML_NS, OLD_ATTS_LOCAL_NAME) != null || currentAtts.getValue(DELTAXML_NS, NEW_ATTS_LOCAL_NAME) != null) {
          // we have a format change (i.e 1 or 3)
          outputExchange();
        } else {
          // we have a pcdata change only
          //if we have format tags, we need to output modified format tags containing a PCDATAmodify, otherwise, just a PCDATAmodify
          if (currentAtts != null) {
            int deltaIndex= currentAtts.getIndex(DELTAXML_NS, DELTA_LOCAL_NAME);
            if (deltaIndex != -1) {
              currentAtts.removeAttribute(deltaIndex);
            }
            //see if we have any att left
            if(currentAtts.getLength() > 0) {
              outputStartTags(currentAtts, deltaIndex != -1);
              outputPCDataModify();
              outputEndTags(currentAtts);
            } else {
              outputPCDataModify();
            }
          } else {
            outputPCDataModify();
          }
        }
      }
    }
    
    unchangedChars= new StringBuffer();
    oldChars= new StringBuffer();
    newChars= new StringBuffer();
  }
  
  private boolean attsEqual(Attributes atts, boolean ignoreDelta) {
    //return true if currentAtts= null or these atts are the same as the currentAtts
    if (currentAtts == null) {
      currentAtts= new AttributesImpl(atts);
      return true;
    } else {
      if (atts.getLength() == currentAtts.getLength()) {
        for (int i= 0; i < atts.getLength(); i++) {
          String qName= atts.getQName(i);
          if(ignoreDelta && qName.equals(DELTAXML_PREFIX + ":" + DELTA_LOCAL_NAME)) {
//            System.out.println("Ignoring a delta att");
            continue;
          }
          String value= atts.getValue(qName);
          String otherValue= currentAtts.getValue(qName);
          if (!value.equals(otherValue)) {
            return false;
          }
        }
        return true;
      } else {
        //different size - must be different
        return false;
      }
    }
  }
  
  private AttributesImpl attStringToAttributes(String attString) {
    AttributesImpl theAtts= new AttributesImpl();
    if(attString != null) {
      int start= 0;
      int index;
      while((index= attString.indexOf('=', start)) != -1) {
        String name= attString.substring(start, index);
        char delimiter= attString.charAt(index+1);
        int end= attString.indexOf(delimiter, index+2);
        String val= attString.substring(index+2, end); //don't actually need this!
        String qName= name;
        String localName;
        String uri= "";
        if (name.indexOf(':') == -1) {
          localName= name;
        } else {
          //we have a prefixed name
          int colonIndex= name.indexOf(':');
          localName= name.substring(colonIndex+1);
          //get the prefix and get it's namespace from the Map
          String prefix= name.substring(0, colonIndex);
          uri= (String)namespaces.get(prefix);
          if (uri == null) {
            System.out.println("WARNING: namespace for prefix '" + prefix + "' could not be found");
            uri= "";
          }
        }
        theAtts.addAttribute(uri, localName, qName, "CDATA", val); 
        start= end+2;
      }
    }
    return theAtts;
  }
  
}
