// $Id: RemoveExchange.java 4699 2008-10-16 14:19:53Z nigelw $
// Copyright (c) 2006 DeltaXML Ltd. All rights reserved

package com.deltaxml.pipe.filters;

import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.AttributesImpl;
import com.deltaxml.pipe.XMLFilterImpl2;

/**
 * <p>Removes deltaxml:exchange elements from the output.</p>
 * <p>More specifically,
 * <pre>
 * &lt;deltaxml:exchange&gt;
 * &nbsp;&nbsp;&lt;deltaxml:old&gt;
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;old-element deltaxml:delta="delete" /&gt;
 * &nbsp;&nbsp;&lt;/deltaxml:old&gt;
 * &nbsp;&nbsp;&lt;deltaxml:new&gt;
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;new-element deltaxml:delta="add" /&gt;
 * &nbsp;&nbsp;&lt;/deltaxml:new&gt;
 * &lt;/deltaxml:exchange&gt;
 * </pre>
 * 
 * is converted to
 * 
 * <pre>
 * &lt;old-element deltaxml:delta="delete" /&gt;
 * &lt;new-element deltaxml:delta="add" /&gt;
 * </pre>
 * 
 * and 
 * 
 * <pre>
 * &lt;deltaxml:exchange&gt;
 * &nbsp;&nbsp;&lt;deltaxml:old&gt;
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;old-element deltaxml:delta="delete" /&gt;
 * &nbsp;&nbsp;&lt;/deltaxml:old&gt;
 * &nbsp;&nbsp;&lt;deltaxml:new&gt;
 * &nbsp;&nbsp;&nbsp;&nbsp;some new text
 * &nbsp;&nbsp;&lt;/deltaxml:new&gt;
 * &lt;/deltaxml:exchange&gt;
 * </pre>
 * 
 * is converted to
 * 
 * <pre>
 * &lt;old-element deltaxml:delta="delete" /&gt;
 * &lt;deltaxml:PCDATAmodify&gt;
 * &nbsp;&nbsp;&lt;deltaxml:PCDATAold /&gt;
 * &nbsp;&nbsp;&lt;deltaxml:PCDATAnew&gt;
 * &nbsp;&nbsp;&nbsp;&nbsp;some new text
 * &nbsp;&nbsp;&lt;/deltaxml:PCDATAnew&gt;
 * &lt;/deltaxml:PCDATAmodify&gt;
 * </pre>
 * </p>
 * @author Tristan Mitchell
 * @version $Revision: 4699 $
 */
public class RemoveExchange extends XMLFilterImpl2
{
  private final String DXML_NS= "http://www.deltaxml.com/ns/well-formed-delta-v1";
  private final String DXML_OLD= "old";
  private final String DXML_NEW= "new";
  private final String DXML_EXCHANGE= "exchange";
  
  private boolean saveText= false;
  
  private String savedText= "";

  /**
   * Constructs a new RemoveExchange filter.
   */
  public RemoveExchange() {
    super();
  }
  
  /**
   * Overrides the default startElement method. This version of the method strips out
   * all exchange-related start tags and determines whether or not to store character events
   * for outputting in a <code>deltaxml:PCDATAmodify</code> element
   */
  public void startElement(String uri, String localName, String qName, Attributes atts)
    throws SAXException 
  {
    
    if (uri.equals(DXML_NS) && localName.equals(DXML_EXCHANGE)) {
      // don't call super.startElement - this element is removed
    } else if (uri.equals(DXML_NS) && localName.equals(DXML_OLD)) {
      saveText= true;
    } else if (uri.equals(DXML_NS) && localName.equals(DXML_NEW)) {
      saveText= true;
    } else {
      // a deltaxml:old or deltaxml:new can either contain text or a single element, if 
      // we find another element we cannot have any text, so saveText is made false
      saveText= false;
      super.startElement(uri, localName, qName, atts);
    }
  }
  
  /**
   * Overrides the default endElement method. This verison of the method strips out
   * all exchange-related end tags and outputs the <code>deltaxml:PCDATAmodify</code>
   * elements if necessary.
   */
  public void endElement(String uri, String localName, String qName) throws SAXException {
    if (uri.equals(DXML_NS) && localName.equals(DXML_EXCHANGE)) {
      // we don't output the exchange close tag either 
    } else if (uri.equals(DXML_NS) && localName.equals(DXML_OLD)) {
      if (saveText) {
        //if we were saving text, we now need to output it in a PCDATAmodify element
        super.startElement(DXML_NS, "PCDATAmodify", "deltaxml:PCDATAmodify", new AttributesImpl());
        super.startElement(DXML_NS, "PCDATAold", "deltaxml:PCDATAold", new AttributesImpl());
        super.characters(savedText.toCharArray(), 0, savedText.length());
        super.endElement(DXML_NS, "PCDATAold", "deltaxml:PCDATAold");
        super.startElement(DXML_NS, "PCDATAnew", "deltaxml:PCDATAnew", new AttributesImpl());
        super.endElement(DXML_NS, "PCDATAnew", "deltaxml:PCDATAnew");
        super.endElement(DXML_NS, "PCDATAmodify", "deltaxml:PCDATAmodify");
        savedText= "";
        saveText= false;
      }
    } else if (uri.equals(DXML_NS) && localName.equals(DXML_NEW)) {
      if (saveText) {
        //if we were saving text, we now need to output it in a PCDATAmodify element
        super.startElement(DXML_NS, "PCDATAmodify", "deltaxml:PCDATAmodify", new AttributesImpl());
        super.startElement(DXML_NS, "PCDATAold", "deltaxml:PCDATAold", new AttributesImpl());
        super.endElement(DXML_NS, "PCDATAold", "deltaxml:PCDATAold");
        super.startElement(DXML_NS, "PCDATAnew", "deltaxml:PCDATAnew", new AttributesImpl());
        super.characters(savedText.toCharArray(), 0, savedText.length());
        super.endElement(DXML_NS, "PCDATAnew", "deltaxml:PCDATAnew");
        super.endElement(DXML_NS, "PCDATAmodify", "deltaxml:PCDATAmodify");
        savedText= "";
        saveText= false;
      }
    } else {
      super.endElement(uri, localName, qName);
    }
  }

  /**
   * Overrides the default characters method. This version of the method saves characters
   * inside a <code>deltaxml:old</code> or <code>deltaxml:new</code> element if necessary.
   */
  public void characters(char [] ch, int start, int length) throws SAXException {
    if (saveText) {
      savedText= savedText.concat(new String(ch, start, length));
    } else {
      super.characters(ch, start, length);
    }
  }  
  
}
